<?php

declare(strict_types = 1);

namespace App\Controller\Admin;

use App\Controller\AppController;
use Cake\Mailer\Email;

/**
 * SchemeRequests Controller
 *
 * @property \App\Model\Table\SchemeRequestsTable $SchemeRequests
 * @method \App\Model\Entity\SchemeRequest[]|\Cake\Datasource\ResultSetInterface paginate($object = null, array $settings = [])
 */
class SchemeRequestsController extends AppController {

    /**
     * Initialize controller
     *
     * @return void
     */
    public function initialize(): void {
        parent::initialize();
        $this->viewBuilder()->setLayout('default-admin');

        $this->Authentication->addUnauthenticatedActions([]);

//        debug($this->Authentication->getResult()->getData());exit;
        if ($this->Authentication->getResult()->getData() && in_array($this->Authentication->getResult()->getData()->users_roles_id, ['cpo', 'pm', 'chief'])) {
            $this->Flash->error(__('Oops, you are not allowed that location'));

            $this->redirect(['controller' => 'Users', 'action' => 'view']);
        }

        //$this->Auth->allow(['index']);
    }

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index() {

        $cond = array();

        if (isset($_GET['status']) && $_GET['status'] !== '') {
            $cond[] = ['SchemeRequests.status like' => '%' . $_GET['status'] . '%'];
        } else {
            $cond[] = ['SchemeRequests.status like' => '%pending%'];
        }
        if (isset($_GET['department'])) {
            $cond[] = ['SchemeRequests.department_id IN' => $_GET['department']];
        }

        $this->paginate = [
            'conditions' => $cond,
            'contain' => ['Departments', 'DepartmentSchemes', 'Users'],
            'order' => ['id' => 'desc']
        ];
        $schemeRequests = $this->paginate($this->SchemeRequests);
        $this->loadModel('Departments');
        $departments = $this->Departments->find('list', ['conditions' => ['deleted' => 0], 'order' => ['department_name' => 'asc']])->all()->toArray();
        $this->set(compact('schemeRequests', 'departments'));
    }

    public function exportpdf($id = null) {
        $this->viewBuilder()->setLayout('');

        $cond = array();
        if (isset($_GET['status']) && $_GET['status'] !== '') {
            $name = ucfirst($_GET['status']);
            $cond[] = ['SchemeRequests.status like' => '%' . $_GET['status'] . '%'];
        } else {
            $cond[] = ['SchemeRequests.status like' => '%pending%'];
            $name = 'Pending';
        }
        if (isset($_GET['department'])) {
            $cond[] = ['SchemeRequests.department_id IN' => $_GET['department']];
        }

        $departmentSchemes = $this->SchemeRequests->find('all', [
                    'conditions' => $cond,
                    'order' => ['SchemeRequests.id' => 'desc'],
                    'contain' => ['Departments', 'DepartmentSchemes'],
//                    'fields' => ['id', 'id_chief_economist', 'overall_status', 'scheme_name', 'scheme_code', 'scheme_type', 'department_id', 'last_modified', 'locked', 'directive', 'mega_project', 'issues_list']
                ])->all();


        $this->set(compact('departmentSchemes', 'name'));
    }

    public function exportall() {
        $output_type = 'D';


        $cond = array();
        if (isset($_GET['status']) && $_GET['status'] !== '') {
            $name = ucfirst($_GET['status']);
            $cond[] = ['SchemeRequests.status like' => '%' . $_GET['status'] . '%'];
        } else {
            $cond[] = ['SchemeRequests.status like' => '%pending%'];
            $name = 'Pending';
        }
        $file = $name . '-Date-Change-Request.xlsx';

        $departmentSchemes = $this->SchemeRequests->find('all', [
                    'conditions' => $cond,
                    'order' => ['SchemeRequests.id' => 'desc'],
                    'contain' => ['Departments', 'DepartmentSchemes'],
//                    'fields' => ['id', 'id_chief_economist', 'overall_status', 'scheme_name', 'scheme_code', 'scheme_type', 'department_id', 'last_modified', 'locked', 'directive', 'mega_project', 'issues_list']
                ])->all();


        $this->set(compact('departmentSchemes', 'output_type', 'file'));

        $this->viewBuilder()->setLayout('xls/default');
        $this->viewBuilder()->setTemplate('xls/exportall');
        $this->RequestHandler->respondAs('xlsx');
        $this->render();
    }

    public function save() {

        if ($this->request->is(['patch', 'post', 'put'])) {
//            echo __LINE__;exit;
            $schemeRequest = $this->SchemeRequests->get($_POST['id']);
            $schemeRequest->request_response = $_POST['request_response'];
            $schemeRequest->status = $_POST['result_status'];
//            debug($schemeRequest);
//            exit;
            if ($this->SchemeRequests->save($schemeRequest)) {
                $this->send_mail($_POST['id'], $_POST['rec_name']);
                echo json_encode('saved');
            } else {
                echo json_encode('not saved');
            }
        }
        exit;
    }

    function send_mail($request_id = null, $rec_name = null) {
//        echo $request_id;
//        var_dump($email);
//        echo $rec_name;
//
//
//        exit;
        $steps_Array = ['schemeprofile' => 'Scheme Profile', 'step1' => 'Is PC-II Required?', 'step2' => 'Is the PC-1 approved?', 'step3' => 'Is Hiring Required?', 'step4' => 'Is Land Acquisition Required?', 'step5' => 'Is Procurement of Goods Required?', 'step6' => 'Is Procurement of Services Required?', 'step7' => 'Is Construction/Civil Work Required?'];


        $this->loadModel('SchemeRequests');
        $schemeRequest = $this->SchemeRequests->get($request_id, [
            'contain' => ['Departments', 'DepartmentSchemes', 'Users'],
        ]);
//debug($schemeRequest);exit;
        $email = new Email();
        $email->viewBuilder()->setTemplate('schemerequest_response', 'default');
        $email->setEmailFormat('html')
                ->setTo($schemeRequest->user->email)
                ->setFrom(['info@cmdu.kp.gov.pk' => 'cmdu.kp.gov.pk'])
                ->setBcc('info@cmdu.kp.gov.pk')
                ->setSubject('cmdu.kp.gov.pk - Date Change Request - Response')
                ->setViewVars(['schemeRequest' => $schemeRequest, 'rec_name' => $rec_name, 'steps_Array' => $steps_Array])
                ->send();

        //sms starts
        $message = 'Dear ' . $rec_name . ', Your date change request Against Scheme: "'
                . ucfirst($schemeRequest->department_scheme->scheme_name) . '" for the step: "' . h($steps_Array[$schemeRequest->step_name])
                . '" is : "' . ucfirst($schemeRequest->status) . '" with remarks: "' . ucfirst($schemeRequest->request_response) . '".';
        $this->loadComponent('Sms');
        if ($schemeRequest->user->tel1) {
            $this->Sms->sms(str_replace(' ', '', $schemeRequest->user->tel1), $message);
            $this->Sms->sms("03339329637", $message);
        }
//        echo $response;
        //sms ends
    }

    /**
     * View method
     *
     * @param string|null $id Scheme Request id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null) {
        $schemeRequest = $this->SchemeRequests->get($id, [
            'contain' => ['Departments', 'SchemeProfiles', 'Users'],
        ]);

        $this->set(compact('schemeRequest'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add() {
        $schemeRequest = $this->SchemeRequests->newEmptyEntity();
        if ($this->request->is('post')) {
            $schemeRequest = $this->SchemeRequests->patchEntity($schemeRequest, $this->request->getData());
            if ($this->SchemeRequests->save($schemeRequest)) {
                $this->Flash->success(__('The scheme request has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The scheme request could not be saved. Please, try again.'));
        }
        $departments = $this->SchemeRequests->Departments->find('list', ['limit' => 200]);
        $schemeProfiles = $this->SchemeRequests->SchemeProfiles->find('list', ['limit' => 200]);
        $users = $this->SchemeRequests->Users->find('list', ['limit' => 200]);
        $this->set(compact('schemeRequest', 'departments', 'schemeProfiles', 'users'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Scheme Request id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null) {
        $schemeRequest = $this->SchemeRequests->get($id, [
            'contain' => [],
        ]);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $schemeRequest = $this->SchemeRequests->patchEntity($schemeRequest, $this->request->getData());
            if ($this->SchemeRequests->save($schemeRequest)) {
                $this->Flash->success(__('The scheme request has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The scheme request could not be saved. Please, try again.'));
        }
        $departments = $this->SchemeRequests->Departments->find('list', ['limit' => 200]);
        $schemeProfiles = $this->SchemeRequests->SchemeProfiles->find('list', ['limit' => 200]);
        $users = $this->SchemeRequests->Users->find('list', ['limit' => 200]);
        $this->set(compact('schemeRequest', 'departments', 'schemeProfiles', 'users'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Scheme Request id.
     * @return \Cake\Http\Response|null|void Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null) {
        $this->request->allowMethod(['post', 'delete']);
        $schemeRequest = $this->SchemeRequests->get($id);
        if ($this->SchemeRequests->delete($schemeRequest)) {
            $this->Flash->success(__('The scheme request has been deleted.'));
        } else {
            $this->Flash->error(__('The scheme request could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

}
